<?php
/*--------------------------------------------------------------
   Repository.php 2021-06-16
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2021 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
 -------------------------------------------------------------*/

declare(strict_types=1);

namespace Gambio\Admin\Modules\Statistics\App\Overview\Data;

use Gambio\Admin\Modules\Statistics\App\Overview\Model\Collections\WidgetDefinitions;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\Collections\Widgets;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\Entities\Widget;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\Entities\WidgetDatabaseEntry;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\Entities\WidgetDefinition;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\Events\Added;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\Events\Configured;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\Events\Deleted;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\Events\Moved;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\Events\Renamed;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\Events\Resized;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\ValueObjects\Height;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\ValueObjects\HorizontalPosition;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\ValueObjects\Id;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\ValueObjects\Label;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\ValueObjects\Option\UpdateSet;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\ValueObjects\Type;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\ValueObjects\VerticalPosition;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\ValueObjects\Width;
use Gambio\Core\Event\Abstracts\AbstractEventDispatchingRepository;
use Gambio\Core\Event\EventDispatcher;

class Repository extends AbstractEventDispatchingRepository
{
    /**
     * @var Reader
     */
    private $reader;
    
    /**
     * @var Writer
     */
    private $writer;
    
    /**
     * @var Mapper
     */
    private $mapper;
    
    
    /**
     * Constructor.
     */
    public function __construct(
        Reader $reader,
        Writer $writer,
        Mapper $mapper,
        EventDispatcher $eventDispatcher
    ) {
        $this->reader = $reader;
        $this->writer = $writer;
        $this->mapper = $mapper;
        
        $this->setEventDispatcher($eventDispatcher);
    }
    
    
    /**
     * Return widgets.
     */
    public function getWidgets(): Widgets
    {
        $databaseEntries = $this->reader->getWidgetDatabaseEntries();
        $definitions     = $this->reader->getWidgetDefinitions();
        
        return $this->mapper->mapToWidgets($databaseEntries, $definitions);
    }
    
    
    /**
     * Return widget by ID.
     */
    public function getWidgetById(Id $id): Widget
    {
        $databaseEntry = $this->getWidgetDatabaseEntryById($id);
        $definition    = $this->reader->getWidgetDefinitionByType($databaseEntry->type());
        
        return $this->mapper->mapToWidget($databaseEntry, $definition);
    }
    
    
    /**
     * Return widget definitions.
     */
    public function getWidgetDefinitions(): WidgetDefinitions
    {
        return $this->reader->getWidgetDefinitions();
    }
    
    
    /**
     * Return widget database entry by ID.
     */
    public function getWidgetDatabaseEntryById(Id $id): WidgetDatabaseEntry
    {
        return $this->mapper->mapToWidgetDatabaseEntry($this->reader->getWidgetDatabaseEntryById($id));
    }
    
    
    /**
     * Return widget definition by type.
     */
    public function getWidgetDefinitionByType(Type $type): WidgetDefinition
    {
        return $this->reader->getWidgetDefinitionByType($type);
    }
    
    
    /**
     * Add widget.
     */
    public function addWidget(WidgetDatabaseEntry $databaseEntry): void
    {
        $this->getWidgetDefinitionByType($databaseEntry->type())
            ->options()
            ->validateUpdateSet($databaseEntry->updateSet()->toArray());
        
        $this->writer->addWidget($databaseEntry);
        $this->dispatchEvent(Added::create($databaseEntry->type()));
    }
    
    
    /**
     * Resize widget.
     */
    public function resizeWidget(Id $id, Width $width, Height $height): void
    {
        $this->writer->resizeWidget($id, $width, $height);
        $this->dispatchEvent(Resized::create($id, $width, $height));
    }
    
    
    /**
     * Move widget.
     */
    public function moveWidget(Id $id, HorizontalPosition $horizontalPosition, VerticalPosition $verticalPosition): void
    {
        $this->writer->moveWidget($id, $horizontalPosition, $verticalPosition);
        $this->dispatchEvent(Moved::create($id, $horizontalPosition, $verticalPosition));
    }
    
    
    /**
     * Configure widget.
     */
    public function configureWidget(Id $id, UpdateSet $updateSet): void
    {
        $databaseEntry = $this->getWidgetDatabaseEntryById($id);
        
        $this->getWidgetDefinitionByType($databaseEntry->type())
            ->options()
            ->validateUpdateSet($databaseEntry->updateSet()->toArray());
        
        $this->writer->configureWidget($id, $updateSet);
        $this->dispatchEvent(Configured::create($id, $updateSet));
    }
    
    
    /**
     * Delete widget.
     */
    public function deleteWidget(Id $id): void
    {
        $this->writer->deleteWidget($id);
        $this->dispatchEvent(Deleted::create($id));
    }
    
    
    /**
     * Rename widget.
     */
    public function renameWidget(Id $id, Label $label): void
    {
        $this->writer->renameWidget($id, $label);
        $this->dispatchEvent(Renamed::create($id, $label));
    }
}